* SetPrinterProperties.prg (c) 2005 Barbara Peisch www.peisch.com
*   You may include or modify any of the code in this program as you see fit.
*   You may not claim it as your own, nor try to enforce a copyright.
*   This program is provided "as is" with no implied guarantees nor waranties.

* Purpose: Programmatically sets printer properties of the public portion
*          of the DEVMODE structure.
* Limitations: Only allows changing of properties from DM_ORIENTATION through
*              DM_COLLATE from the Windows.h file

* Created on August 27, 2005

* Dependencies:
*   VFP 9 or higher

* Usage:
*  In order to use this class, you must know the constants from the Windows.h 
*  file that you wish to change, plus the constants for the settings.

*  1. Instantiate the class
*  2. Call SetProperty
*  3. Check for errors
*  4. Either store the DEVMODE returned or print your reports
*  5. Call ResetPrinter                                         (Optional)
*  6. Change more properties                                    (Optional)
*  7. Either store the DEVMODE returned or print your reports   (Optional)
*  8. Release the object reference

*********************************************************
*!*	* Example 1 - Using the current printer

*!*	SET PRINTER TO NAME (GETPRINTER())
*!*	loSetPrinterProperties = CREATEOBJECT('SetPrintProperties')
*!*	loSetPrinterProperties.SetProperty(DM_ORIENTATION, DMORIENT_LANDSCAPE)
*!*	IF loSetPrinterProperties.lError
*!*		MESSAGEBOX(loSetPrinterProperties.cError)
*!*	ELSE 
*!*		REPORT FORM Report1 TO PRINTER
*!*		REPORT FORM Report2 TO PRINTER 
*!*	ENDIF 
*!*	loSetPrinterProperties = NULL

*********************************************************
*!*	* Example 2 - Storing the DEVMODE returned

*!*	loSetPrinterProperties = CREATEOBJECT('SetPrintProperties')
*!*	lcDevmode = loSetPrinterProperties.SetProperty(DM_DEFAULTSOURCE, DMBIN_MANUAL, 'MyPrinter')
*!*	llSuccess = not loSetPrinterProperties.lError
*!*	lcError = loSetPrinterProperties.cError
*!*	loSetPrinterProperties = NULL
*!*	IF llSuccess
*!*		* Do something here with lcDevmode
*!*	ELSE 
*!*		MESSAGEBOX(lcError)
*!*	ENDIF 

*********************************************************
*!*	* Example 3 - Changing multiple settings, resetting and changing more settings

*!*	SET PRINTER TO NAME (GETPRINTER())
*!*	loSetPrinterProperties = CREATEOBJECT('SetPrintProperties')
*!*	loSetPrinterProperties.SetProperty(DM_ORIENTATION, DMORIENT_LANDSCAPE)
*!*	IF loSetPrinterProperties.lError
*!*		MESSAGEBOX(loSetPrinterProperties.cError)
*!*	ELSE 
*!*		loSetPrinterProperties.SetProperty(DM_DEFAULTSOURCE, DMBIN_MANUAL, 'MyPrinter')
*!*		IF loSetPrinterProperties.lError
*!*			MESSAGEBOX(loSetPrinterProperties.cError)
*!*		ELSE 
*!*			REPORT FORM Report1 TO PRINTER
*!*			REPORT FORM Report2 TO PRINTER 

*!*			loSetPrinterProperties.ResetPrinter()
*!*			loSetPrinterProperties.SetProperty(DM_COPIES, 5)
*!*			IF loSetPrinterProperties.lError
*!*				MESSAGEBOX(loSetPrinterProperties.cError)
*!*			ELSE 
*!*				loSetPrinterProperties.SetProperty(DM_DUPLEX, DMDUP_VERTICAL)
*!*				IF loSetPrinterProperties.lError
*!*					MESSAGEBOX(loSetPrinterProperties.cError)
*!*				ELSE 
*!*					REPORT FORM Report3 TO PRINTER
*!*					REPORT FORM Report4 TO PRINTER 
*!*				ENDIF 
*!*			ENDIF 
*!*		ENDIF 
*!*	ENDIF 
*!*	loSetPrinterProperties = NULL

***********************************************************************************

DEFINE CLASS SetPrintProperties as Custom 

lError = .F.
cError = ''

cOrigDevmode  = ''
cOrigDevnames = ''
cOrigExpr     = ''
cOrigPrinter  = ''
cPrinter2Use  = ''
cNewDevmode   = ''
cPrevDevmode  = ''

	*--------------------------------------------------------------------
	FUNCTION Init()
	LOCAL lcRptFile, lnOldSelect 
	
	lnOldSelect = SELECT()
	
	* Use a unique file name so we can use this in a multi-user situation
	* Using a cursor instead of a physical file doesn't work, but we can
	* create the FRX from a cursor.
	lcRptFile = SYS(2015)+".FRX"
	CREATE CURSOR TempCur (Temp C (10))
	CREATE REPORT (JUSTSTEM(lcRptFile)) FROM TempCur
	USE IN TempCur
	USE (lcRptFile) EXCLUSIVE ALIAS RptFile
	SYS(1037,2)
	This.cOrigExpr     = EXPR
	This.cOrigDevnames = TAG
	This.cOrigDevmode  = TAG2
	This.cNewDevmode   = TAG2
	This.cOrigPrinter  = SET("Printer",3)
	This.cPrevDevmode  = ''

	* Get rid of the temporary FRX
	USE IN RptFile
	ERASE (JUSTSTEM(lcRptFile)+".*")

	SELECT (lnOldSelect)

	ENDFUNC 
	*--------------------------------------------------------------------
	FUNCTION Destroy
	
	This.ResetPrinter()
	
	ENDFUNC 
	*--------------------------------------------------------------------
	FUNCTION ResetPrinter
	LOCAL lcRptFile, lnOldSelect 
	
	lnOldSelect = SELECT()
	
	* Use a unique file name so we can use this in a multi-user situation
	* Using a cursor instead of a physical file doesn't work, but we can
	* create the FRX from a cursor.
	lcRptFile = SYS(2015)+".FRX"
	CREATE CURSOR TempCur (Temp C (10))
	CREATE REPORT (JUSTSTEM(lcRptFile)) FROM TempCur
	USE IN TempCur
	USE (lcRptFile) EXCLUSIVE ALIAS RptFile
	replace Expr WITH This.cOrigExpr, ;
			Tag  WITH This.cOrigDevnames, ;
			Tag2 WITH This.cOrigDevmode
	SYS(1037,3)
	This.cNewDevmode = TAG2

	* Get rid of the temporary FRX
	USE IN RptFile
	ERASE (JUSTSTEM(lcRptFile)+".*")

	SELECT (lnOldSelect)
	
	SET PRINTER TO NAME (This.cOrigPrinter)
	
	ENDFUNC 
	*--------------------------------------------------------------------
	FUNCTION SetProperty(tnPropertyPos, tnSetting, tcPrinter)
	* Sets the property positioned in the tcPropertyPos of the DEVMODE
	* structure to the setting in tcSetting.
	* Will act on the printer named in tcPrinter, or the current printer if tcPrinter is empty
	* These parameters must values representing constants from the Windows.H file
	* Will return the DEVMODE structure or an empty string if there's an error

	LOCAL lcPrinter, lcTemp
	
	* Validate parameters
	IF TYPE('tnPropertyPos') <> 'N' OR TYPE('tnSetting') <> 'N'
		This.lError = .T.
		This.cError = "You must send values for the first two parameters"
		RETURN ''
	ENDIF 

	This.lError = .F.
	This.cError = ''
	IF EMPTY(tcPrinter)
		lcPrinter = SET("Printer",3)
	ELSE
		lcPrinter = tcPrinter
		SET PRINTER TO NAME (lcPrinter)
	ENDIF 
	
	* Done with parameter validation.
	
	This.DoIt(tnPropertyPos, tnSetting)
	RETURN This.cNewDevmode

	ENDFUNC 
	*--------------------------------------------------------------------
	FUNCTION DoIt(tnPropertyPos, tnSetting)
	LOCAL lcDevmode, lnFlags, lcRptFile, lnOldSelect, lnPropOffs, lnDC, lnBuf, lcFac
	
	IF EMPTY(This.cNewDevmode)
		* It may be empty from an error on the previous attempt to change something
		This.cNewDevmode = This.cPrevDevmode
	ENDIF 
	
	lnFlags = CTOBIN(SUBSTR(This.cNewDevmode,41,4),'4sr')
	IF BITAND(lnFlags,tnPropertyPos) <> 0
		lnPropOffs = This.GetProp4DM(tnPropertyPos,'Offset')
		IF lnPropOffs <> 0
			* Validate that the setting requested is supported for this printer.
			* Note: In some cases this isn't necessary because the check against
			* the flags field in DEVMODE indicated whether or not the property is
			* available, e.g. Duplex, Collate.  The DeviceCapabilities would just
			* return the same info.  Right now, the only check it does is for
			* paper bins.
			lnDC = This.GetProp4DM(tnPropertyPos,"DC")
			DO CASE 
			CASE lnDC < 0
				* Property Position not supported
				RETURN 
			CASE lnDC = 0
				* Not sure how to validate
			OTHERWISE
				lnBuf = This.GetProp4DM(tnPropertyPos,"Buf")
				lcFac = This.GetProp4DM(tnPropertyPos,"Fac")
				* Confirm that the setting is available for the printer
				IF (lnBuf <> 0 OR NOT EMPTY(lcFac)) AND NOT This.CheckDC(lnDC, tnSetting, lnBuf, lcFac)
					RETURN 
				ENDIF 
			ENDCASE 

			This.cNewDevmode = STUFF(This.cNewDevmode, lnPropOffs, 2, BINTOC(tnSetting,'2sr'))

			* Write the changes to the printer
			lnOldSelect = SELECT()
			
			* Use a unique file name so we can use this in a multi-user situation
			* Using a cursor instead of a physical file doesn't work, but we can
			* create the FRX from a cursor.
			lcRptFile = SYS(2015)+".FRX"
			CREATE CURSOR TempCur (Temp C (10))
			CREATE REPORT (JUSTSTEM(lcRptFile)) FROM TempCur
			USE IN TempCur
			USE (lcRptFile) EXCLUSIVE ALIAS RptFile

			replace expr WITH '', ;
					tag  WITH '', ;
					tag2 WITH This.cNewDevmode

			SYS(1037,3)		&& Writes the printer settings out to the printer

			* Get rid of the temporary FRX
			USE IN RptFile
			ERASE (JUSTSTEM(lcRptFile)+".*")

			SELECT (lnOldSelect)
		ELSE
			This.lError = .T.
			This.cError = "Invalid property value sent as first parameter"
			This.cNewDevmode = ''
		ENDIF 
	ELSE
		This.lError = .T.
		This.cError = "That property not available for the printer"
		This.cPrevDevmode = This.cNewDevmode
		* Blank out the Devmode so we return an empty string
		This.cNewDevmode = ''
	ENDIF 

	ENDFUNC 
	*--------------------------------------------------------------------
	FUNCTION GetProp4DM(tnPropertyPos, tcProperty)
	* Returns either the offset or the corresponding DeviceCapability value
	* tnPropertyPos is the value of the bit for the property in the dmFields
	* tcProperty is either "Offset" or "DC", or "Buf"
	LOCAL lnOffset, lnDCValue, lnBuf, lvReturn, lcFactor

	* The offset is the offset in the DEVMODE structure for the property
	* The DCValue is the contant for the device capabilities from Windows.h
	* lnBuf is the size of the output buffer from DeviceCapabilities
	* lcFac is the "factor" setting for DeviceCapabilities
	*       "Buf" means to multiply the number of entries returned by the
	*             initial call to DeviceCapabilities by the value of lnBuf
	*       "Set" means to check the return value directly against the
	*             setting without doing a CTOBIN conversion
	DO CASE 
	CASE tnPropertyPos = 1			&& DM_ORIENTATION
		lnOffset = 45
		lnDCValue = 0
		lnBuf = 0
		lcFactor = ''
	CASE tnPropertyPos = 2			&& DM_PAPERSIZE
		lnOffset = 47
		lnDCValue = 2	&& DC_PAPERS
		lnBuf = 2
		lcFactor = 'Buf'
	CASE tnPropertyPos = 4			&& DM_PAPERLENGTH
		lnOffset = 49
		lnDCValue = 3	&& DC_PAPERSIZE
		lnBuf = 0
*		lcFactor = 'PointY'
		lcFactor = ''
	CASE tnPropertyPos = 8			&& DM_PAPERWIDTH
		lnOffset = 51
		lnDCValue = 3	&& DC_PAPERSIZE
		lnBuf = 0
*		lcFactor = 'PointX'
		lcFactor = ''
	CASE tnPropertyPos = 16			&& DM_SCALE
		lnOffset = 53
		lnDCValue = 0
		lnBuf = 0
		lcFactor = ''
	CASE tnPropertyPos = 256		&& DM_COPIES
		lnOffset = 55
		lnDCValue = 18	&& DC_COPIES
		lnBuf = 0
		lcFactor = 'Set'
	CASE tnPropertyPos = 512		&& DM_DEFAULTSOURCE
		lnOffset = 57
		lnDCValue = 6	&& DC_BINS
		lnBuf = 2
		lcFactor = 'Buf'
	CASE tnPropertyPos = 1024		&& DM_PRINTQUALITY
		lnOffset = 59
		lnDCValue = 13	&& DC_ENUMRESOLUTIONS
*		lnBuf = 64
		lnBuf = 0
		lcFactor = 'Buf'
	CASE tnPropertyPos = 2048		&& DM_COLOR
		lnOffset = 61
		lnDCValue = 0
		lnBuf = 0
		lcFactor = ''
	CASE tnPropertyPos = 4096		&& DM_DUPLEX
		lnOffset = 63
		lnDCValue = 0
		lnBuf = 0
		lcFactor = ''
	CASE tnPropertyPos = 8192		&& DM_YRESOLUTION
		lnOffset = 65
		lnDCValue = 13	&& DC_ENUMRESOLUTIONS
		lnBuf = 0
		lcFactor = 'PointY'
	CASE tnPropertyPos = 16384		&& DM_TTOPTION
		lnOffset = 67
		lnDCValue = 0
		lnBuf = 0
		lcFactor = ''
	CASE tnPropertyPos = 32768		&& DM_COLLATE
		lnOffset = 69
		lnDCValue = 0
		lnBuf = 0
		lcFactor = ''
	OTHERWISE
		This.lError = .T.
		This.cError = "This program doesn't support the property position requested."
		lnOffset = 0
		lnDCValue = -1
		lnBuf = 0
		lcFactor = ''
	ENDCASE 

	DO case
	CASE UPPER(tcProperty) = "DC"
		lvReturn = lnDCValue
	CASE UPPER(tcProperty) = "OFFSET"
		lvReturn = lnOffset
	CASE UPPER(tcProperty) = "BUF"
		lvReturn = lnBuf
	CASE UPPER(tcProperty) = "FAC"
		lvReturn = lcFactor
	OTHERWISE
		lvReturn = 0
	ENDCASE 

	RETURN lvReturn

	ENDFUNC 
	*--------------------------------------------------------------------
	FUNCTION CheckDC(tnValue, tnSetting, tnBuf, tcFactor)
	* tnValue = Constant that represents the capability to be checked
	* tnSetting = The setting to use for the capability
	* tnBuf = The size of the output buffer returned by DeviceCapabilities for this capability
	* tcFactor = What technique to use for multiplying/checking
	*            "BUF" = Use tnBuf
	*            "SET" = Use straight setting
	*            "POINTX" = Use X setting of the POINT structure
	*            "POINTY" = Use Y setting of the POINT structure
	LOCAL lnIndex, lcPrinterName, lnEntries, lcPort, lcIdBfr, lnDCCnt, lnDCID

	DECLARE INTEGER DeviceCapabilities IN winspool.drv;
	  STRING pDevice, STRING pPort, INTEGER fwCapability,;
	  STRING @pOutput, INTEGER pDevMode

	DECLARE INTEGER GetLastError IN kernel32

	lcPrinterName = SET("Printer",3)
	APRINTERS(laAvailPrinters)
	FOR i = 1 TO ALEN(laAvailPrinters,1)
		IF UPPER(laAvailPrinters[i,1]) = UPPER(lcPrinterName)
			lcPort = laAvailPrinters[i,2]
			EXIT 
		ENDIF 
	ENDFOR

	* First call DeviceCapabilities in a way that returns the size of the buffer we need
	lnEntries = DeviceCapabilities(lcPrinterName, lcPort, tnValue, 0, 0)
	IF lnEntries = -1
		This.ErrDC()
		RETURN .F.
	ELSE
		DO CASE 
		CASE UPPER(tcFactor) = "BUF"
			lcIDBuf = REPLICATE(CHR(0), lnEntries * tnBuf)
		CASE UPPER(tcFactor) = "SET"
			* Assume that we can't have an output of more than 7 digits
			lcIDBuf = REPLICATE(CHR(0), 7)
		CASE UPPER(LEFT(tcFactor,5)) = "POINT"
			lcIDBuf = REPLICATE(CHR(0), 64 * lnEntries)
		ENDCASE 

		lnDCCnt = DeviceCapabilities(lcPrinterName, lcPort, tnValue, @lcIdBuf, 0)
		IF lnDCCnt = -1
			This.ErrDC()
		ELSE 
			DO CASE 
			CASE UPPER(tcFactor) = "BUF"
				FOR i = 1 TO lnDCCnt
					if CTOBIN(SUBSTR(lcIdBuf, (i*2)-1, 2),'2sr') = tnSetting
						RETURN .T.
					ENDIF
				ENDFOR
			CASE UPPER(tcFactor) = "SET"
				IF lnDCCnt >= tnSetting
					RETURN .T.
				ENDIF  
			CASE UPPER(tcFactor) = "POINTX"
			CASE UPPER(tcFactor) = "POINTY"
				FOR i = 1 TO lnDCCnt
					if CTOBIN(SUBSTR(lcIdBuf, (i*2), 2),'2sr') = tnSetting
						RETURN .T.
					ENDIF
				ENDFOR
			ENDCASE 

			This.lError = .T.
			This.cError = "That setting not supported"
			RETURN .F.
		ENDIF 
	ENDIF 

	ENDFUNC 
	*--------------------------------------------------------------------
	FUNCTION ErrDC
	LPARAMETERS tnErr
	LOCAL lnLastErr
	* Code from Calvin
	
	#define FORMAT_MESSAGE_ALLOCATE_BUFFER 0x00000100
	#define FORMAT_MESSAGE_IGNORE_INSERTS  0x00000200
	#define FORMAT_MESSAGE_FROM_STRING     0x00000400
	#define FORMAT_MESSAGE_FROM_HMODULE    0x00000800
	#define FORMAT_MESSAGE_FROM_SYSTEM     0x00001000
	#define FORMAT_MESSAGE_ARGUMENT_ARRAY  0x00002000
	#define FORMAT_MESSAGE_MAX_WIDTH_MASK  0x000000FF
	 
	DECLARE integer GetSystemDefaultLangID IN win32api
	 
	DECLARE integer FormatMessage IN WIN32API integer,integer,integer,integer,string,integer
	langid=BITAND(0xffff,GetSystemDefaultLangID())
	 
	lnLastErr = GetLastError()
	cstr=SPACE(1024)
	nlen=FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 0,lnLastErr,langid, @cstr, LEN(cstr))
	IF nlen > 0
		this.lError = .T.
		This.cError = LEFT(cstr,nlen)
	ENDIF 

	ENDFUNC 
	*--------------------------------------------------------------------

ENDDEFINE 
***********************************************************************************
